"use client";
import postData from "@/api/post-data";
import { Button } from "@/components/ui/button";
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormMessage,
} from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { toastError, toastSuccess } from "@/services/custom-toast";
import { zodResolver } from "@hookform/resolvers/zod";
import { Loader2, Send } from "lucide-react";
import { useLocale, useTranslations } from "next-intl";
import CountUp from "react-countup";
import { useForm } from "react-hook-form";
import { z } from "zod";
import CustomBreadcrumb from "../ui/shared/CustomBreadcrumb";
import FallbackImage from "../ui/shared/FallbackImage";
import section1 from "/public/about/section1.png";
import section2 from "/public/about/section2.png";

const counter = [
  {
    id: 1,
    name: "order-in-month",
    count: 19905,
  },
  {
    id: 2,
    name: "daily-user",
    count: 535,
  },
  {
    id: 3,
    name: "product-added",
    count: 4235,
  },
];

export default function AboutUs() {
  const t = useTranslations("about-us");
  const label = useTranslations("form-labels");
  const error = useTranslations("form-errors");
  const locale = useLocale();
  const form = useForm({
    defaultValues: {
      message: "",
      email: "",
      name: "",
    },
    resolver: zodResolver(
      z.object({
        message: z
          .string({ errorMap: () => ({ message: error("required") }) })
          .min(1),
        name: z
          .string({ errorMap: () => ({ message: error("required") }) })
          .min(1),
        email: z
          .string({ errorMap: () => ({ message: error("email") }) })
          .email(),
      })
    ),
  });

  const {
    setError,
    formState: { isSubmitting },
  } = form;

  async function onSubmit(data) {
    const formData = new FormData();
    formData.append("message", data.message);
    formData.append("name", data.name);
    formData.append("email", data.email);
    const { error, message, errors } = await postData({
      endPoint: "/contact-us",
      body: formData,
      locale,
    });
    if (error) {
      for (const [key, value] of Object.entries(errors)) {
        // Loop in through errors
        setError(
          key,
          {
            type: "server",
            message: value,
          },
          {
            shouldFocus: true,
          }
        );
      }
      toastError(message); // Display error message
      return;
    }
    toastSuccess(message);
    form.reset();
  }
  return (
    <>
      <CustomBreadcrumb />
      {/* Text & Images */}
      <section className="container mb-24 gap-8 lg:gap-16 grid grid-cols-1 md:grid-cols-2">
        {/* Section 1 */}
        {/* Image */}
        <div className="w-full col-span-1">
          <FallbackImage src={section1} alt="section1" />
        </div>
        {/* Text */}
        <div className="w-full col-span-1 text-main mt-6 max-lg:mb-12">
          <h1 className="font-bold text-xl mb-4 md:text-2xl lg:text-4xl">
            {t("intro")}
          </h1>
          <h2 className="font-bold text-xl mb-6 lg:mb-12 md:text-2xl lg:text-4xl">
            {t("title")}
          </h2>
          <p
            className="text-sm text-justify md:text-base mb-12 md:mb-8 lg:mb-24"
            style={{
              lineHeight: "2.4rem",
            }}
          >
            {t("content")}
          </p>

          {/* Counters */}
          <div className="w-full justify-between flex">
            {counter.map(({ count, name }, i) => (
              <div key={i}>
                <CountUp
                  className="text-main font-extrabold mb-4 text-3xl lg:text-5xl"
                  end={count}
                  prefix="+"
                />
                <p className="text-main text-center text-base">
                  {t(`counter.${name}`)}
                </p>
              </div>
            ))}
          </div>
        </div>
        {/* Section1 End */}

        {/* Section 2 */}
        {/* Text */}
        <div className="w-full text-main col-span-1">
          <span className="font-bold text-xl mb-4 md:text-2xl">
            {t("intro2")}
          </span>
          <h1 className="font-bold mb-6 lg:mb-12 text-xl md:text-2xl lg:text-4xl">
            {t("title2")}
          </h1>
          <p
            className="mb-10 lg:mb-20 text-justify text-sm md:text-base"
            style={{
              lineHeight: "2.4rem",
            }}
          >
            {t("content")}
          </p>
          <p
            className="text-sm text-justify md:text-base"
            style={{
              lineHeight: "2.4rem",
            }}
          >
            {t("content")}
          </p>
        </div>
        {/* Image */}
        <div className="w-full col-span-1">
          <FallbackImage src={section2} alt="section2" />
        </div>
      </section>

      {/* <Testimonial /> */}

      {/* Form */}
      <section className="bg-main  mt-20 mb-24  p-12">
        <div className="text-center text-white/80 space-y-4 mb-12">
          <h1 className="font-bold text-2xl lg:text-4xl">{t("question")}</h1>
          <p>{t("feel-free")}</p>
        </div>
        <Form {...form}>
          <form
            method="post"
            onSubmit={form.handleSubmit(onSubmit)}
            className="flex gap-4 items-start justify-center max-md:flex-col"
          >
            <Button
              disabled={isSubmitting}
              type="submit"
              className="bg-[#D6DBEC] gap-4 hidden md:inline-flex hover:bg-gray-300 hover:text-black lg:w-44 text-main"
            >
              {isSubmitting ? <Loader2 className="animate-spin" /> : <Send />}

              {label("send")}
            </Button>
            <FormField
              name="name"
              control={form.control}
              render={({ field }) => (
                <FormItem>
                  <FormControl>
                    <Input
                      className="lg:w-72"
                      {...field}
                      placeholder={label("name")}
                    />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />
            <FormField
              name="email"
              control={form.control}
              render={({ field }) => (
                <FormItem>
                  <FormControl>
                    <Input
                      className="lg:w-72"
                      {...field}
                      placeholder={label("email")}
                    />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />
            <FormField
              name="message"
              control={form.control}
              render={({ field }) => (
                <FormItem>
                  <FormControl>
                    <Input
                      className="lg:w-72"
                      {...field}
                      placeholder={label("question")}
                    />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />
            <Button
              disabled={isSubmitting}
              type="submit"
              className="bg-[#D6DBEC] gap-4 inline-flex md:hidden hover:bg-gray-300 hover:text-black  text-main"
            >
              {isSubmitting ? <Loader2 className="animate-spin" /> : <Send />}
              {label("send")}
            </Button>
          </form>
        </Form>
      </section>
    </>
  );
}
