import fetchData from "@/api/fetch-data";
import {
  Pagination,
  PaginationContent,
  PaginationItem,
  PaginationLink,
  PaginationNext,
  PaginationPrevious,
} from "@/components/ui/pagination";
import clsx from "clsx";
import { getLocale, getTranslations } from "next-intl/server";
import { GoArrowLeft } from "react-icons/go";
import AlertWarning from "../ui/shared/AlertWarning";
import BlogCardRedesignThree from "../ui/shared/BlogCardRedesignThree/BlogCardRedesignThree";
import CustomBreadcrumb from "../ui/shared/CustomBreadcrumb";


export default async function Articles({ searchParams }) {
  const currentPage = parseInt(searchParams?.page || '1', 10);
  const t = await getTranslations(); // Retrieve translations


  // Fetch data for the current page
  const res = await fetchData({
    endPoint: `/articles?page=${currentPage}`,
  });

  // Handle error or empty data
  if (res?.error) {
    return (
      <AlertWarning text={res.message || "Service currently unavailable"} />
    );
  }

  if (res?.data?.articles?.length === 0) {
    return <AlertWarning text={t("nodata")} />;
  }

  const { data, status } = res;
  const totalPages = data.pagination.total_pages;


  return (
    <>
      {/* Navigation Header */}
      <CustomBreadcrumb />

      {/* Article Cards */}
      <section className="container mt-4 lg:mt-8 mb-12 grid gap-4 grid-cols-1 md:grid-cols-2 lg:gap-8 lg:grid-cols-4">
        {status === "success" &&
          data.articles.map((item) => (
            <div key={item.id}>
              <BlogCardRedesignThree blog={item} />
            </div>
          ))}
      </section>
      {/* Pagination */}
      {
        totalPages > 1 &&
        <Pagination dir="ltr" className="mb-14 space-x-2">
          {/* Previous Button */}
          {currentPage == 1 ? <div
            className="rounded-full border p-0  flex items-center justify-center opacity-75 border-dark"
            style={{ width: "2.5rem", height: "2.5rem" }}
            disabled={currentPage == 1}
          >
            <GoArrowLeft className="size-5 font-normal" />
          </div> :
            <PaginationPrevious
              className="rounded-full border p-0 border-dark"
              href={`?page=${currentPage + 1}`}
              style={{ width: "2.5rem", height: "2.5rem" }}
              disabled={currentPage == 1}
            />}
          {/* Pagination Links */}
          <PaginationContent>
            {Array.from({ length: totalPages }).map((_, index) => (
              <PaginationItem key={index}>
                <PaginationLink
                  href={`?page=${index + 1}`}
                  style={{ width: "2.5rem", height: "2.5rem" }}

                  className={clsx("rounded-full bg-white border", {
                    "bg-main text-white hover:bg-main hover:text-white":
                      index + 1 === currentPage,
                  })}
                >
                  {index + 1}
                </PaginationLink>
              </PaginationItem>
            ))}
          </PaginationContent>

          {/* Next Button */}
          <PaginationNext
            className="rounded-full border p-0 border-dark  shrink-0"
            href={`?page=${currentPage + 1}`}
            style={{ width: "2.5rem", height: "2.5rem" }}
            disabled={currentPage == totalPages}
          />
        </Pagination>
      }
    </>
  );
}
