"use client";

import SectionHeader from "@/app/ui/shared/SectionHeader";
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar";
import { Card, CardContent, CardHeader } from "@/components/ui/card";
import {
   Carousel,
   CarouselContent,
   CarouselItem,
} from "@/components/ui/carousel";
import clsx from "clsx";
import { useLocale, useTranslations } from 'next-intl';
import { useEffect, useState } from 'react';
import avatar from "/public/about/avatar.png";

const testimonials = [
   {
      id: 1,
      comment: "This product is amazing! It has changed my life for the better.",
      author: "Ahmed Mohamed",
      rating: 5,
      avatar: ""
   },
   {
      id: 2,
      comment: "Excellent customer service and high product quality.",
      author: "Fatima Ali",
      rating: 4,
      avatar: ""
   },
   {
      id: 3,
      comment: "Wonderful experience, I will definitely recommend it to my friends.",
      author: "Mohamed Hassan",
      rating: 5,
      avatar: ""
   },
   {
      id: 4,
      comment: "Good product but there's room for improvement.",
      author: "Sarah Ahmed",
      rating: 3,
      avatar: ""
   },
   {
      id: 5,
      comment: "Reasonable price for the value provided.",
      author: "Omar Khaled",
      rating: 4,
      avatar: ""
   }
];


/**
 * @author: Ahmed Badran
 * Created At: 23/10/2024
 * A component that renders a testimonial section with a carousel
 *
 * @remarks
 * The component uses the Carousel component from the `@/components/ui/carousel` module
 * and the Avatar component from the `@/components/ui/avatar` module. It also uses the
 * `useLocale` hook from the `next-intl` module and the `useTranslations` hook from the
 * `next-intl` module.
 *
 * The component expects an array of testimonials as a prop. Each testimonial should be
 * an object with the following properties: `id`, `comment`, `author`, `rating`, and
 * `avatar`. The `avatar` property should be an object with the following properties: `src`
 * and `alt`.
 *
 * The component renders a section with a title and a carousel of testimonials. The
 * title is rendered as an h1 element with a font size of 2xl and a bold font weight. The
 * title is split into two parts: the first part is rendered as a span element with a
 * font size of 2xl and a bold font weight, and the second part is rendered as a span
 * element with a font size of 2xl and a bold font weight and a border bottom of 4px.
 *
 * The carousel is rendered as a div element with a class name of `space-x-6 lg:space-x-12`
 * and a class name of `select-none`. The carousel contains a list of carousel items, each
 * of which is rendered as a div element with a class name of `md:basis-1/2 lg:basis-1/3`
 * and a class name of `text-white bg-dark rounded-2xl`. Each carousel item contains a
 * card with a header and a content area. The header contains an avatar and a heading
 * with a font size of xl and a bold font weight. The content area contains a paragraph
 * with a font size of base and a normal font weight.
 *
 * The component also renders a navigation bar with a class name of `flex justify-center mt-4 gap-2 overflow-x-auto pb-2`
 * and a class name of `size-2 rounded-sm bg-gray-400 transition-all duration-300 flex-shrink-0`. The navigation bar
 * contains a list of buttons, each of which is rendered as a button element with a class
 * name of `size-2 rounded-sm bg-gray-400 transition-all duration-300 flex-shrink-0`. Each
 * button has a click event handler that scrolls to the corresponding testimonial when
 * clicked.
 *
 * @param {object[]} testimonials - An array of testimonials
 * @returns {ReactElement} - A React element that renders the testimonial section
 */
export default function Testimonial() {
   const [api, setApi] = useState();
   const [currentSlide, setCurrentSlide] = useState(0);
   const [totalSlides, setTotalSlides] = useState();
   const locale = useLocale();
   const t = useTranslations('testimonial');
   const [intro, title] = t('title').split(' ');

   useEffect(() => {
      if (!api) {
         return;
      }
      setTotalSlides(api.slideNodes().length);
      api.on("select", () => {
         setCurrentSlide(api.selectedScrollSnap())
      })
   }, [api])

   return (
      <section>
         <div className="container mb-16">
            <SectionHeader title={intro} marker={title} withLink={false} />
         </div>
         <Carousel
            className="w-full select-none"
            setApi={setApi}
            dir="ltr"
         >
            <CarouselContent className="space-x-6 lg:space-x-12">
               {testimonials.map((testimonial) => (
                  <CarouselItem dir={locale === 'ar' ? 'rtl' : 'ltr'} key={testimonial.id} className="md:basis-1/2 lg:basis-1/3 text-white bg-dark rounded-2xl">
                     <Card className="bg-inherit border-none text-inherit">
                        <CardHeader className="flex flex-row items-center gap-4">
                           <Avatar className="size-14">
                              <AvatarImage src={avatar.src} alt={`${testimonial.author}'s photo`} />
                              <AvatarFallback>{testimonial.author.slice(0, 2)}</AvatarFallback>
                           </Avatar>
                           <h2 className="font-bold text-xl lg:text-2xl">{t('name')}</h2>
                        </CardHeader>
                        <CardContent className="text-white/80 font-normal">
                           {t('comment')}
                        </CardContent>
                     </Card>
                  </CarouselItem>
               ))}
            </CarouselContent>
            <div className="flex justify-center mt-4 gap-2 overflow-x-auto pb-2">
               {Array.from({ length: totalSlides }).map((_, index) => (
                  <button
                     key={index}
                     className={clsx(`size-2 rounded-sm bg-gray-400 transition-all duration-300 flex-shrink-0`, {
                        'bg-main h-2 w-8': index === currentSlide,
                     })}
                     onClick={() => api?.scrollTo(index)}
                  />
               ))}
            </div>
         </Carousel>
      </section>
   );
}

